"use client";

import Link from "next/link";
import { usePathname } from "next/navigation";
import { Button } from "@/components/ui/button";
import { Locale } from "@/lib/i18n";
import { Menu, X, Globe, ChevronDown } from "lucide-react";
import { useState } from "react";
import { motion, AnimatePresence } from "framer-motion";
import { cn } from "@/lib/utils";

interface NavbarProps {
  lang: Locale;
  dict: any;
}

export function Navbar({ lang, dict }: NavbarProps) {
  const [isOpen, setIsOpen] = useState(false);
  const pathname = usePathname();

  const toggleMenu = () => setIsOpen(!isOpen);

  // Helper to switch language but keep path (simple version)
  const switchLang = (newLang: string) => {
    const segments = pathname.split('/');
    segments[1] = newLang; // Assuming /lang/rest-of-path
    return segments.join('/');
  };

  return (
    <header className="fixed top-0 left-0 right-0 z-50 bg-white/80 backdrop-blur-md border-b border-gray-100">
      <div className="container mx-auto px-4 md:px-6 h-16 flex items-center justify-between">
        {/* Logo */}
        <Link href={`/${lang}`} className="flex items-center gap-2 font-bold text-xl tracking-tighter">
          <div className="w-8 h-8 rounded-lg bg-brand-primary flex items-center justify-center text-white">
            R
          </div>
          <span>RSaSh Lab</span>
        </Link>

        {/* Desktop Nav */}
        <nav className="hidden md:flex items-center gap-8 text-sm font-medium text-text-secondary">
          <Link href={`/${lang}/#products`} className="hover:text-brand-primary transition-colors">
            {dict.products}
          </Link>
          <Link href={`/${lang}/#services`} className="hover:text-brand-primary transition-colors">
            {dict.services}
          </Link>
           <Link href={`/${lang}/#tech`} className="hover:text-brand-primary transition-colors">
            {dict.technologies}
          </Link>
          <Link href={`/${lang}/about`} className="hover:text-brand-primary transition-colors">
            {dict.about}
          </Link>
          <Link href={`/${lang}/contact`} className="hover:text-brand-primary transition-colors">
            {dict.contact}
          </Link>
        </nav>

        {/* Actions */}
        <div className="hidden md:flex items-center gap-4">
           {/* Lang Switcher - Simplified */}
           <div className="flex items-center gap-2 text-xs font-semibold text-text-secondary">
              <Link href={switchLang('en')} className={cn("hover:text-brand-primary", lang === 'en' && "text-brand-primary")}>EN</Link>
              <span className="text-gray-300">|</span>
              <Link href={switchLang('ru')} className={cn("hover:text-brand-primary", lang === 'ru' && "text-brand-primary")}>RU</Link>
              <span className="text-gray-300">|</span>
              <Link href={switchLang('uz')} className={cn("hover:text-brand-primary", lang === 'uz' && "text-brand-primary")}>UZ</Link>
           </div>
           
           <Link href={`/${lang}/contact`}>
             <Button size="sm">
              {dict.getStarted}
             </Button>
           </Link>
        </div>

        {/* Mobile Toggle */}
        <button className="md:hidden p-2 text-text-secondary" onClick={toggleMenu}>
          {isOpen ? <X /> : <Menu />}
        </button>
      </div>

      {/* Mobile Menu */}
      <AnimatePresence>
        {isOpen && (
          <motion.div 
            initial={{ opacity: 0, y: -20 }}
            animate={{ opacity: 1, y: 0 }}
            exit={{ opacity: 0, y: -20 }}
            className="absolute top-16 left-0 right-0 bg-white border-b border-gray-100 p-4 md:hidden shadow-lg"
          >
            <nav className="flex flex-col gap-4 text-base font-medium">
                <Link href={`/${lang}/#products`} onClick={toggleMenu}>{dict.products}</Link>
                <Link href={`/${lang}/#services`} onClick={toggleMenu}>{dict.services}</Link>
                <Link href={`/${lang}/#tech`} onClick={toggleMenu}>{dict.technologies}</Link>
                <Link href={`/${lang}/about`} onClick={toggleMenu}>{dict.about}</Link>
                <Link href={`/${lang}/contact`} onClick={toggleMenu}>{dict.contact}</Link>
                <hr className="my-2"/>
                <div className="flex gap-4">
                   <Link href={switchLang('en')} className={cn(lang === 'en' && "text-brand-primary")}>EN</Link>
                   <Link href={switchLang('ru')} className={cn(lang === 'ru' && "text-brand-primary")}>RU</Link>
                   <Link href={switchLang('uz')} className={cn(lang === 'uz' && "text-brand-primary")}>UZ</Link>
                </div>
                <Link href={`/${lang}/contact`} className="w-full">
                  <Button className="w-full">{dict.getStarted}</Button>
                </Link>
            </nav>
          </motion.div>
        )}
      </AnimatePresence>
    </header>
  );
}
